/**
* \file: VideoChannel.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include "VideoChannel.h"
#include "utils/Statistics.h"

namespace adit { namespace carplay
{

using namespace std;

VideoChannel::VideoChannel()
{
    session = nullptr;
    frame = nullptr;
    videoMutex = PTHREAD_MUTEX_INITIALIZER;

}

VideoChannel::~VideoChannel()
{
    LOGD_DEBUG((dipo, "VideoChannel stopped"));
}

void VideoChannel::FirstFrameRendered()
{
    session=Session::Get(nullptr);
    if(session != nullptr)
    {
        session-> VideoPlaybackStarted();
    }
    else
    {
        LOG_ERROR((dipo,"No active session "));
    }
}

bool VideoChannel::Initialize(Session* session, IConfiguration* config)
{
    this->session = session;

    auto ptr = Factory::Instance()->Create(config->GetItem("IVideoOutAdapter", ""));
    mainScreen = move(unique_ptr<IVideoOutAdapter>(static_cast<IVideoOutAdapter*>(ptr)));
    if (mainScreen != nullptr)
    {
        int width = config->GetNumber("display-width", 800);
        int height = config->GetNumber("display-height", 480);
        int fps = config->GetNumber("display-fps", 30);
        if (!mainScreen->Initialize(*config, width, height, (IVideoOutAdapter::Framerate)fps,*this, static_cast<SessionId>(this->session)))
        {
            // TODO error handling
            LOG_ERROR((dipo, "IVideoOutAdapter::Init"));
            return false;
        }
    }
    else
    {
        // TODO error handling
        LOG_ERROR((dipo, "no IVideoOutAdapter implementation!"));
        return false;
    }

    return true;
}

bool VideoChannel::Start()
{
    if (mainScreen != nullptr)
    {
        pthread_mutex_lock(&videoMutex);
        mainScreen->Start();
        pthread_mutex_unlock(&videoMutex);
        Statistics::Instance().SetAVChannel(session, AVChannel_Video, true);

    }
    currentFrameValid = true;
    isNewframe = true;
    try
    {
        frame = new FrameInternal();
        return true;
    }
    catch(std::bad_alloc& exc)
    {
        LOG_ERROR((dipo, "Failed to create current object while starting the videochannel!"));
        return false;
    }

}

void VideoChannel::Stop()
{
    if (mainScreen != nullptr)
    {
        pthread_mutex_lock(&videoMutex);
        mainScreen->Stop();
        pthread_mutex_unlock(&videoMutex);
        if(frame != nullptr)
        {
            delete(frame);
        }
        Statistics::Instance().SetAVChannel(session, AVChannel_Video, false);
    }
}

void VideoChannel::RenderScreen(const uint8_t* inData, size_t inLen, uint64_t inDisplayTicks,
        CFDictionaryRef inOptions, ScreenStreamCompletion_f inCompletion, void* inContext)
{
    (void)inOptions;

    if (mainScreen == nullptr)
    {
        LOG_WARN_RATE(60, (dipo, "no IVideoOut instance"));
        if (inCompletion != nullptr)
            inCompletion(inContext);
        currentFrameValid = false;
        return;
    }
    if(currentFrameValid)
    {
        if (isNewframe)
        {
            frame->SetDisplayTime(inDisplayTicks);
            isNewframe = false;
        }
        else
        {
            size_t reqBufsize = GetNextPowerOfTwo(frame->GetLength() + inLen);
            if(frame->GetBufSize() != reqBufsize)
            {
                if(!frame->CreateNewBuff(reqBufsize))
                {
                    LOG_ERROR((dipo, "Failed to allocate memory for video frame!"));
                    if (inCompletion != nullptr)
                        inCompletion(inContext);
                    currentFrameValid = false;
                    return;
                }

            }

        }

        pthread_mutex_lock(&videoMutex);
        frame->Append(inData,inLen);
        pthread_mutex_unlock(&videoMutex);

        if (inCompletion != nullptr)
            inCompletion(inContext);

        currentFrameValid = true;
    }
}

void VideoChannel::Finish()
{
    pthread_mutex_lock(&videoMutex);
    if(frame != nullptr && currentFrameValid)
    {
        mainScreen->Push(*frame);
    }
    else
    {
        currentFrameValid = true;
        LOG_WARN((dipo, "The last Frame not rendered to Screen, since Frame Memory pointer is Null!"));
    }

    pthread_mutex_unlock(&videoMutex);

    isNewframe = true;
}
size_t VideoChannel::GetNextPowerOfTwo(size_t inLength)
{
    size_t count = 0;

    if (inLength && !(inLength&(inLength-1)))
    {
        return(inLength);
    }
    while( inLength != 0)
    {
        inLength  >>= 1;
        count += 1;
    }
    return(1<<count);

}
} } // namespace adit { namespace carplay
